/***************************************************************************
                          cmrilib.c  -  description
                             -------------------
    begin                : Thu Jun 17 2004
    copyright            : (C) 2004 by Jason Nishiyama
    email                : evilscientist@shaw.ca
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "cmrilib.h"

	/* function: cmri_make_initialize
		 purpose: initializes SUSIC/USIC/SMINI nodes
		 inputs:
		 	By value:
		 		cardtype : single ascii character, N for no special features, X for SUSIC with 32 bit cards, M for SMINI
		 		nn			 : positive integer node number
		 		dl			 : unsigned integer delay constant
		 		ni			 : positive integer number of input ports (always 3 for SMINI)
		 		no			 : positive integer number of output ports (always 6 for SMINI)
		 		ns			 : integer number of signals for SMINI, number of card sets for SUSIC/USIC
		 	By reference:
		 		*ct		   : pointer to array of card sets/signals. array length should be the same ns
		 outputs: returns 0 on success, appropriate error code on failure
		 created:17 June 2004
		 last modified:22 June 2004
		 Other routines needed:cmri_countsigs, cmri_createstring, cmri_countports, cmri_txdata
		*/
 int cmri_make_initialize (char cardtype, int nn, unsigned int dl, int ni, int no, int ns, int *ct)
{
	char *plaininfo,*sendinfo;
	unsigned char ctbyte;
	int	charpos, loop, loop2, bad, end, count, sendlen, error;
	
	plaininfo=(char *) malloc(256);  /* set aside memory for strings */
	sendinfo=(char *) malloc(256);
	
	/* time to start creating the initialization string*/
	
	charpos=0; /* set string pointer to start of string */
	
	/* first character is the node number */
	if ((nn<0)||(nn>127))  /* check if invalid */
	{
		free (plaininfo);
		free(sendinfo);
		return INVALID_NODE_NUMBER;
	}
	else
		plaininfo[charpos++]=(char)(nn&0xff)+65;
	
	/* this will be an initialization message, so second char will be "I" */
	plaininfo[charpos++]='I';
	
	/* the node definition parameter comes next, we need to check to see if it is valid */
	switch(cardtype)
	{
		case 'N':	plaininfo[charpos++]='N'; /* SUSIC/USIC card */
				break;
						
		case 'X':	plaininfo[charpos++]='X'; /* SUSIC with 32 bit cards*/
				break;
							
		case 'M':	plaininfo[charpos++]='M'; /* SMINI */
				break;
											
		default:	free(plaininfo);
				free(sendinfo);
				return INVALID_NODE_DESCRIPTOR; /* invalid node description, return error */
		
	}
	
	/* next comes the delay length DL. This is a big-endian number so we will have to save it as such */
	plaininfo[charpos++]=(char)((dl&0xff00)>>4); /* high byte of dl */
	plaininfo[charpos++]=(char)(dl&0xff); /* low byte of dl */
	
	/* time to check that ns is valid, since we've checked the node descriptor already, no need to do it again. */
	
	if (cardtype=='M') /* this is an SMINI */
	{
		/* ok, only a specific set of SMINI checks in this section */

		/* check for valid number of output ports, SMINI's all have 6 */
		if (no!=6)
		{
			free (plaininfo);
			free (sendinfo);
			return INVALID_NUMBER_OUTPORT;
		}
		
		/* check for valid number of input ports, SMINI's all have 3*/
		if (ni!=3)
		{
			free (plaininfo);
			free (sendinfo);
			return INVALID_NUMBER_INPORT;
		}

		
		/* we'll put the number of signals into the string first, since the routine will stop if an invalid number is entered*/
		plaininfo[charpos++]=(char) (ns&0xff);
		
		/* we only have to check signal validity if there are signals reported */
		if (ns>0)
		{		
			loop=0; /* set up our loop through *ct*/
			count=0; /* clear our signal counter*/
		
			while (loop<6) /* get count of signals, and check for port validity */
			{
				ctbyte=(char)(ct[loop++]&0xff);  /* get data from ct*/
				loop2=0; /*set check counter*/
				bad=1;	/* set bad flag, assume illegal value */			
				while(loop2<34) /* 24 elements in legal signal array */
				{
					if (ctbyte==cmri_leg_sig[loop2++])	/* check for legal value*/
						bad=0;											/* it's legal, clear flag*/
				}
				if(bad) /* check for illegal signal value*/			
				{
					free(plaininfo);
					free(sendinfo);
					return ILLEGAL_SIGNAL_VALUE;
				}
				/* ok, the signal value is legal, lets count the number of signals for the total.*/
				count+=cmri_countsigs(ctbyte);	
			}		
			/* we now have the number of signals and have checked that the bit values are legal, now to check if we were lied to*/
			if (count!=ns) /* does the number of signals reported equal those shown */
			{
					free(plaininfo);
					free(sendinfo);
					return SIGNAL_NUMBER_MISMATCH;
			}
		
			/*signal information has passed checking, add it to the string*/
			loop=0; /* set up loop */
			while (loop<6) /*loop for number of out ports on SMINI*/
			{
				plaininfo[charpos++]=(char)(ct[loop++]&0xff);
			}
		}
		/* done SMINI initialization string checking and building*/
	}
	else
	{
		/* every other type of card is checked here */
		/*First, check if there are too many cards*/
		if (ns>16)
		{
			free(plaininfo);
			free(sendinfo);
			return TOO_MANY_CARDS;
		}
		/* we also must have at least one card */
		if (ns<1)
		{
			free(plaininfo);
			free(sendinfo);
			return NOT_ENOUGH_CARDS;
		}

		/* ok enough cards, now check for validity
		 * This is a two step process. Since only the last card set can have blanks, we must first check
		 * the first, non-blank groups, then the blank group at the end.
		 */
		
		 loop=0;/* set counting loop */
		 count=0; /* reset port counter */
		 while (loop<ns)
		 {
		 	/* is this the last set, if so we use a different check start and end point */
		 	if ((loop+1)==ns)
		 	{
		 		loop2=0;  /* begining of card list, contains blank cards */
		 		end=14;   /* end of blank cards in card list*/
		 	}
		 	else
		 	{
		 		loop2=14;	/* begining of card list where non-blank cards stored*/
		 		end=30;		/* end of list */
		 	}
		 	ctbyte=(unsigned char) (ct[loop++]&0xff);		 	/* get card info to test */
		 	
		 	bad=1;   /* assume all bad */
		 	while(loop2<end)
		 	{
				if (ctbyte==cmri_leg_card[loop2++]) /* check vs table */
					bad=0;  /* if ok, reset bad flag */
		 	}
					count+=cmri_countports(ctbyte); /* count the number of cards */
		 	
		 	/* if bad still set, not a valid card value */
		 	if (bad)
		 	{
		 		free(plaininfo);
		 		free(sendinfo);
		 		return INVALID_CARD_VALUE;
		 	}
		 }	
		
		 /* ok, now to check if the number of ports reported is the same */
		 if (cardtype=='N')
		 	{/* ok 24 bit cards */
		 		if ((count*3)!=(ni+no))
		 		{
		 			free(plaininfo);
		 			free(sendinfo);
		 			return INVALID_PORT_NUMBER;
		 		}
		 	}	
		 if (cardtype=='X')
		 	{/* ok 32 bit cards */
		 		if ((count*4)!=(ni+no))
		 		{
		 			free(plaininfo);
		 			free(sendinfo);
		 			return INVALID_PORT_NUMBER;
		 		}
		 	}
		 /* ok valid number of cards and ports*/				 		
		 /* time to build string */
		 /* add ns (num cards) to string */
		 plaininfo[charpos++]=(char) (ns&0xff);
		 loop=0; /* loop counter clear*/
		 while (loop<ns)
		 	plaininfo[charpos++]=(char) (ct[loop++]&0xff); /* put ct data into string */
		 	
		 /* done SUSIC/USIC processing */		
	}
	
	/*time to get the string for output*/
	sendlen=0;
	cmri_createsendstring(plaininfo,charpos,sendinfo,&sendlen);
	error=cmri_txdata(sendinfo, sendlen); /* send the data */
	if (error) /* something went wrong*/
	{
		free(plaininfo);
		free(sendinfo);
		return error;
	}
		
	/* ok successful initialization!*/
	
	free(plaininfo);
	free(sendinfo);
	return NO_ERROR;	
}


	/* function: cmri_make_transmit
		 purpose: creates a string to send a transmit-data message, then sends it.
		 inputs:
		 				By value:
		 						nn		 :int node number
		 						no     :number of elements to output
		 				By reference:
		 					*ct		   : int pointer to output data
		 	returns:0 if no errors, appropriate error message on error.				
		 	created:21 June 2004
		 	last modified:21 June 2004
		 	Other routines needed: cmri_createstring
		*/
	
int cmri_make_transmit(int nn, int no, int *ct)
{

	char *plaininfo, *sendinfo;
	int loop,outlen,charpos, error;
	
	plaininfo=(char *) malloc(256); /* set aside memory for our strings */
	sendinfo=(char *) malloc(256);
	
	charpos=0; /* our pointer to where in the plain string we are*/

	/* first character is the node number */
	if (nn>127)  /* check if invalid */
	{
		free(plaininfo);
		free(sendinfo);
		return INVALID_NODE_NUMBER;
	}
	else
		plaininfo[charpos++]=(char)(nn&0xff)+65;
	
	plaininfo[charpos++]='T'; /* indicate a transmit-data message */
	
	loop=0; /*clear our loop counter*/
	while (loop<no) /* loop till all of ct array done*/
	{
		plaininfo[charpos++]=(char)(ct[loop++]&0xff); /* save the data into the string */
	}

	outlen=0;    /*clear output length*/
	cmri_createsendstring(plaininfo,charpos,sendinfo,&outlen); /* make an output string */
	
	error=cmri_txdata(sendinfo, outlen); /* send the data */
	if (error) /* something went wrong*/
	{
		free(plaininfo);
		free(sendinfo);
		return error;
	}
	
	free(plaininfo);
	free(sendinfo);
	return NO_ERROR;	
}	

	/* function: cmri_make_poll
		 purpose: creates a string to send a poll message, then sends it.
		 inputs:
		 				By value:
		 						nn		 :int node number
		 	returns:0 if no errors, appropriate error message on error.				
		 	created:21 June 2004
		 	last modified:21 June 2004
		 	Other routines needed: cmri_createstring
		*/

int cmri_make_poll(int nn)
{
	char *plaininfo, *sendinfo;
	int outlen,charpos,error;
	
	plaininfo=(char *) malloc(256); /* set aside memory for our strings */
	sendinfo=(char *) malloc(256);
	
	charpos=0; /* our pointer to where in the plain string we are*/

	/* first character is the node number */
	if (nn>127)  /* check if invalid */
	{
		free(plaininfo);
		free(sendinfo);
		return INVALID_NODE_NUMBER;
	}
	else
		plaininfo[charpos++]=(char)(nn&0xff)+65;
	
	plaininfo[charpos++]='P'; /* indicate a poll message */

	outlen=0;    /*clear output length*/
	cmri_createsendstring(plaininfo,charpos,sendinfo,&outlen); /* make an output string */

	error=cmri_txdata(sendinfo, outlen); /* send the data */
	if (error) /* something went wrong*/
	{
		free(plaininfo);
		free(sendinfo);
		return error;
	}
	
	/* all good!*/
	free(plaininfo);
	free(sendinfo);
	return NO_ERROR;	
	
}
	
	/* function: cmri_createsendstring
		 purpose: takes the string given, adds start framing, DLE and end framing information
		 inputs:
		 				By value:
		 					lin			: length of input string
		 				By reference:
		 					*in		   : char pointer to output data
		 					*out		 : char pointer to output string
		 					*lout    : length of output string
		 	outputs: returns 0 on success, lout set to lenght of output string.
		 	created:17 June 2004
		 	last modified:17 June 2004
		 	Other routines needed: none
		*/


int cmri_createsendstring (char *in, int lin, char *out, int *lout)
{
	int inpointer,outpointer,slen;
	
	slen=lin; /* get the length of the string */
	
	/* put sync bytes and STX in output string */
	sprintf(out,"\xFF\xFF\x02");
	
	/* ok, now it's time to populate the data portion of the string*/
	
	inpointer=0;  /* point to the start of the input string */
	outpointer=3; /* point to the first free char of the output string*/
	
	while (inpointer<slen)
	{
		/* as we go through the input string, we have to check to handle any DLE sequences */
		/* DEL sequences are needed on 0x02, 0x10, and 0x03 */
		
		switch(in[inpointer])
		{
			case 0x02:			out[outpointer++]=0x10;  /* add DLE */
											break;
											
			case 0x03:      out[outpointer++]=0x10;  /* add DLE */
											break;
											
			case 0x10:      out[outpointer++]=0x10;  /* add dle */
											break;
		}
		
		/* ok, we've put in the DLE if needed, now just copy over the character*/
		out[outpointer++]=in[inpointer++];
	}

		/* at this point we have the data in the string, now to append the ETX and NULL characters */
		out[outpointer++]=0x03; /* ETX */

		*lout=outpointer;
				
		return NO_ERROR;	
}


	/* function: cmri_countsigs
		 purpose: determines the number of two lead signals from bit locations
		 inputs:
						By Value:
						c			:byte (char) containing signal information		 					
		 	outputs: returns number of signals
		 	created:17 June 2004
		 	last modified:18 June 2004
		 	Other routines needed: none
		*/


int cmri_countsigs(char c)
{
	int counter,total;
	char cpy,test;
	
	cpy=c; /* get local copy of c */
	counter=0;	/* clear counter */
	total=0;	/* clear our bit counter */
	
	while (counter<8) /* 8 bits in a byte, 7 shifts to get them all */
	{
		test=cpy&0x01; /* mask off first bit */
		if (test)   /* check if bit set */
			total++;  /* set, count it*/
		cpy=cpy>>1; /* shift to next bit */
		counter++;  /* increment loop counter */
	}
	
	total=total/2; /* #of signals is bits/2*/
	
	return total; /* return the number of signals */
}

	/* function: cmri_countports
		 purpose: determines the number of ports bit locations
		 inputs:
						By Value:
						c			:byte (char) containing port information		 					
		 	outputs: returns number of ports
		 	created:18 June 2004
		 	last modified:18 June 2004
		 	Other routines needed: none
		*/


int cmri_countports(char c)
{
	int counter,total;
	char cpy,test;
	
	cpy=c; /* get local copy of c */
	counter=0;	/* clear counter */
	total=0;	/* clear our bit counter */
	
	while (counter<8) /* 8 bits in a byte, 7 shifts to get them all */
	{
		test=cpy&0x01; /* mask off first bit */
		if (test)   /* check if bit set */
			total++;  /* set, count it*/
		cpy=cpy>>1; /* shift to next bit */
		counter++;  /* increment loop counter */
	}
	
	return total; /* return the number of ports */
}


	/* function: cmri_printstring
		 purpose: debuging routine, used to print out string of serial data
		 inputs:
						By Value:
						slen			:integer - length of string in bytes
						otype                   :character - type of output h-hex, d-decimal, o-octal
						By Reference:
						s					:pointer to char - containing string to be printed
		 	outputs: returns 0
		 	created:18 June 2004
		 	last modified:16 January 2005: add choice of output base
			              12 January 2005:send output to stderr
		 	Other routines needed: none
		*/


int cmri_printstring(char *s, int slen, char otype)
{
	int loop,length;
	
	length=slen;
	loop=0;
	while(loop<length)
	  switch(otype)
	    {
	    case 'h':		fprintf(stderr,"%X ",s[loop++]);
	      break;
	    
	    case 'd':           fprintf(stderr,"%d ",s[loop++]);
	      break;

	    case 'o':           fprintf(stderr,"%o ",s[loop++]);
	      break;

	    default:            fprintf(stderr,"%d ",s[loop++]);


	    }

	printf("\n");
	return NO_ERROR;
}

	/* function: cmri_txdata
		 purpose: takes the string given and transmits it to the globally defined output port
		 inputs:
		 				By value:
		 					slen			: length of input string
		 				By reference:
		 					*s		   : char pointer to output data
		 				Globals:
		 					cmri_sport_id : port handle
		 	outputs: returns 0 on success, appropriate error message on error
		 	created:21 June 2004
		 	last modified:08 July 2004
		 	Other routines needed: none
		*/

int cmri_txdata(char *s, int slen)
{

	int loop;

	
	if (cmri_sport_id==-1) /* port not open */	
		return SERIAL_PORT_NOT_OPEN;
	
	loop=0;	/* clear our loop counter */
	while (loop<slen)
	{
		if(write(cmri_sport_id,&s[loop++],1)!=1) /* write a byte, check if it all went ok*/
		{
			/* something went wrong */
			return SERIAL_WRITE_ERROR;
		}
	}
	
	/* made it!*/
	
	return NO_ERROR;
}


	/* function: cmri_rxdata
		 purpose: takes the string given and transmits it to the globally defined output port
		 inputs:
		 				By value:
		 					tout		 : integer time out value, value in milliseconds.
		 				By reference:
		 					*s		   : char pointer to input data
		 					*slen		 : int pointer to length of data
		 			  Globals:
		 			  	cmri_sport_id : port handle
		 	outputs: returns 0 on success, appropriate error message on error, *s contains input data stripped of formating but with the first byte being the node number, *slen conatins length of *s
		 	created:24 June 2004
		 	last modified:16 June 2005
			                 Fix dle processing
			              08 July 2004
		 	Other routines needed: cmri_getbyte
		*/


int cmri_rxdata(char *s, int *slen, int tout)
{		

	int done,count,error, dleflag;
	unsigned char charin;
	
	
	count=0; /* clear string pointer*/
	done=0; /*clear done flag*/
	
	while (done<2) /* we're going to loop until two sync bytes or by timeout or ETX */
	{
		/*First thing to do is to grab our sync bytes 0xff 0xff*/
		error=cmri_getbyte(&charin,tout);


		if (error)
		{
			/* ok, some error occured*/
			done=4;	
			return error;
		}
	
		if (charin!=0xff)  /* framing error */
		{
			done=4;
			return SERIAL_READ_FRAME_ERROR;
		}
		done++;		/* ok to next byte */
		
	}

	/*next char should be the STX*/
	
	error=cmri_getbyte(&charin, tout);

	if (error)  /* something went wrong*/
		return error;

	if (charin!=0x02) /* not an STX?*/
		return SERIAL_READ_FRAME_ERROR;

	/*next char should be the node number + 0x41*/
	
	error=cmri_getbyte(&charin, tout);


	if (error)  /* something went wrong*/
		return error;

	s[count++]=charin-0x41; /*save the node number*/

       	/*next char should be the "R" character*/
	
	error=cmri_getbyte(&charin, tout);


	if (error)  /* something went wrong*/
		return error;

	if (charin!=0x52) /* not an STX?*/
		return SERIAL_READ_FRAME_ERROR;
		
	
	/* ok get rest of data */
	done=0; /*clear done flag*/
	while (!done)
	{
	  dleflag=0;

		error=cmri_getbyte(&charin, tout);  /* get next byte*/

		if(error)		/* uhoh - another error */
			return error;

		if ((charin==0x10)) /* if dle char  then this is a dle, set flag, get next char... */
		  {
		    error=cmri_getbyte(&charin, tout);  /* get next byte*/

		    if(error)		/* uhoh - another error */
			return error;
		    s[count++]=charin;
		    
		  }
		else /* not a dle char*/
		  {

		  switch (charin)
		    {
		    case 0x02:	        return SERIAL_READ_FRAME_ERROR; /* shouldn't get STX at this point */
		      break;
		      
		    case 0x10:		return UNSPECIFIED_ERROR;			/* shouldn't get DLE here */
		      break;			
		      
		    case 0x03:		done=1;												/* end of text, done */
		      break;
		      
		    default:			s[count++]=charin;						/* just a character, save it*/
		    }
							
		}

	
	}
	
	/* ok we've got the string, save string lenght and get out of here...*/
	*slen=count;
	
	
	return NO_ERROR;
	
	
}		
	/* function: cmri_open_serial_port
		 purpose: Opens the selected serial port with the selected properties
		 inputs:
		 				By value:
		 					port	: serial port to use, see notes
							baud	: baud rate to use, see notes
							usb	: use usb to rs232 interface, see notes
		 				Global used:
		 					cmri_sport_id : handle for serial port.
		 	outputs: returns 0 on success, appropriate error message on error
		 	created:22 June 2004
		 	last modified:12 Jan 2005
			                 remove cannocal input processing
			              30 Nov 2004
					added USB support

		 	Other routines needed: none
		
		Notes:
			The routine currently supports 4 ports:ttyS0, ttyS1, ttyS2 and ttyS3 corresponding
			to DOS' COM1, COM2, COM3 and COM4 which in turn correspond to values of port of 0, 1, 2 and 3.		
		
			Baud rates are 0=9600, 1=19200, 2=57600 and 3=115200. Note that baud rates of 28800 are not
			supported by the termios routines.
			
			the usb flag was added to support the use of usb to rs232 converters. 0=standard serial port, 1=usb serial port
		
		*/

int cmri_open_serial_port(int port, int baud)
{
	char *portspec;
	struct termios options;
	
	/* first check for data in valid ranges */
	
	if ((port<0)||(port>12)) /* port value right out of range!*/
	{
		return INVALID_SERIAL_PORT;
	}
	
	if ((port>3)&&(port<10)) /* between traditional serial ports and usb ports*/
	{
		return INVALID_SERIAL_PORT;
	}
	
	
	if ((baud<0)||(baud>3)) /* Is baud rate in valid range */
	{
		return INVALID_BAUD_RATE;
	}
	
	/* user input is good, lets go on */
		
	portspec=(char *) malloc(50);  /* set aside memory for the port info*/
	
	if (port>3)
		sprintf(portspec,"/dev/ttyUSB%d",(port-10)); /* set up port id for USB*/
	else
		sprintf(portspec,"/dev/ttyS%d",port); /* set up port id for serial port*/

	cmri_sport_id=open(portspec,O_RDWR|O_NOCTTY|O_NDELAY); /* open the port */

	free(portspec); /* don't need this anymore, give the memory back */	
		
	if (cmri_sport_id==-1)   /* check if it worked */
	{
		/* port could not be opened */
		return UNABLE_TO_OPEN_PORT;
	}
	/* ok the port was opened successfully */
	
	/* clear flags*/
	fcntl(cmri_sport_id,F_SETFL,0);
	
	/* ok, now time to set up baud rate, etc...*/
	
	/* get the current options for the port*/
	tcgetattr(cmri_sport_id, &options);
	
	/* set the baudrate to what the user asked for */
	cfsetispeed(&options,cmri_baud[baud]);
	cfsetospeed(&options,cmri_baud[baud]);
	
	/* enable receiver and set local mode */
	options.c_cflag |= (CLOCAL | CREAD);

	/* set bits for 8N2 */
	options.c_cflag &=~PARENB; 	/* NO PARITY */
	options.c_cflag &=~CSTOPB;	/* clear stop bits flag*/
	options.c_cflag |=CSTOPB;		/* set 2 stop bits*/
	options.c_cflag &=~CSIZE;		/* clear size bits*/
	options.c_cflag |=CS8;			/* set 8 bits*/
	
	/* set canocal input */
	options.c_lflag =0; /* non cannocal, no echo... etc...*/

	/* no output processing */
	options.c_oflag &= ~OPOST;


	/* no parity checking on input*/
	options.c_iflag &=~INPCK;
	/* we also want all 8 bits*/
	options.c_iflag &=~ISTRIP;
	
	/* ok, good to send */
	tcsetattr(cmri_sport_id,TCSANOW,&options);
	
	/* set no blocking on no char */
	fcntl(cmri_sport_id,F_SETFL,FNDELAY);
	
	/* done, port set up*/
	return NO_ERROR;
		
}

	/* function: cmri_sighandler
		 purpose: handles the timer signal by setting the timeout flag
		 inputs:
			By value:
				sig	: signal trapped
			Global used:
				cmri_timeout_sig : timeout flag
	 	outputs: none
	 	created:25 June 2004
	 	last modified:25 June 2004
	 	Other routines needed: none
		
		Notes:
				routine will be called when the SIGALRM is asserted and will set the timeout flag
		*/



void cmri_sighandler(int sig)
{
	if (sig==SIGALRM)  /* check for real timer alarm */
		cmri_timeout_sig=1;
}

	/* function: cmri_filltime
		 purpose: fills the itimerval structure with the provided data
		 inputs:
			By value:
				val_sec	: it.value.tv._sec
				val_usec : it.value.tv_usec
				int_sec : it.interval.tv_sec
				int_usec : it.interval.tv_usec
			By reference:
				timevalue : pointer to itimerval structure
	 	outputs: returns 0
	 	created:25 June 2004
	 	last modified:25 June 2004
	 	Other routines needed: none
		
		*/

int cmri_filltime(struct itimerval *timevalue,int val_sec, int val_usec, int int_sec, int int_usec)
{
	timevalue->it_value.tv_sec=val_sec; /* place data into structure */
	timevalue->it_value.tv_usec=val_usec;
	timevalue->it_interval.tv_sec=int_sec;
	timevalue->it_interval.tv_usec=int_usec;
	
	return NO_ERROR;
}


	/* function: cmri_readbyte
		 purpose: gets a byte from the serial port
		 inputs:
			By value:
				tout : time out delay in microseconds
			By reference:
				c :pointer to single character to store recieved byte
			Global used:
				cmri_timeout_sig : timeout flag
				cmri_sport_id : serial port handle
	 	outputs: returns character in c and 0 on success, appropriate error message on failure
	 	created:08 July 2004
	 	last modified:12 January 2004
		                  disable timer when completed.

	 	Other routines needed: cmri_sighandler
		
		*/


int cmri_getbyte(char *c, int tout)
{
	int sec,usec;	
	struct sigaction act, oldact;
	struct itimerval timevalue, oldvalue;
	
	if (cmri_sport_id==-1) /* port not open */	
		return SERIAL_PORT_NOT_OPEN;
	
	/*first thing, set up the timeout signal handler*/
	
	cmri_timeout_sig=0;		/* clear the flag variable */
	act.sa_handler=&cmri_sighandler; /* point to the signal handler*/
	act.sa_flags=0; /* clear flags */
	sigaction(SIGALRM,&act,&oldact); /* set the signal handler to activate when the real itimer runs out */

	/* now work out how much time before timeout */
	sec=tout/1000; /*get seconds from timeout*/
	usec=(tout-(sec*1000))*1000; /* get microseconds left over once seconds removed */
	cmri_filltime(&timevalue,sec,usec,0,0); /* put data into structure */
	cmri_filltime(&oldvalue,0,0,0,0); /* clear storeage structure */


/* clear timeout flag*/

	cmri_timeout_sig=0;
	
	setitimer(ITIMER_REAL,&timevalue,&oldvalue); /* start timeout timer*/

		while ((read(cmri_sport_id,c,1)==-1)&&(cmri_timeout_sig==0)); /* wait for input or timeout */
		if (cmri_timeout_sig)		/* we timed out*/
		{	
			
			/* return the signal handler to its original state*/
			sigaction(SIGVTALRM,&oldact,&act);
		
			return SERIAL_READ_TIMEOUT; /* return timeout error*/
		}
	
		/*disable timer*/
		cmri_filltime(&timevalue,0,0,0,0); /* fill structure with zeros to disable timer*/
		setitimer(ITIMER_REAL,&timevalue,&oldvalue);

		/* return the signal handler to its original state*/
				sigaction(SIGALRM,&oldact,&act);

		return NO_ERROR;  /*no error, return*/
	
}
